/*____________________________________________________________________________
		Copyright (C) 2000 Network Associates, Inc.
        All rights reserved.

        $Id: CWindow.cpp,v 1.4 2000/02/01 01:37:51 nryan Exp $
____________________________________________________________________________*/

#include "pgpClassesConfig.h"

#include "CString.h"
#include "UMath.h"

#include "CAccelerator.h"
#include "CDC.h"
#include "CRect.h"
#include "CSize.h"
#include "CWindow.h"
#include "UModuleInstance.h"

_USING_PGP

// Class CWindow static variables

PGPUInt32			CWindow::mWindowMapRefCount	= 0;
CHashTable<CWindow>	*CWindow::mWindowMap;
CRMWOLock			*CWindow::mWindowMapLock;


// Class CWindow member functions

CWindow::CWindow() : 

	mWeCreated(FALSE), mWeSubclassed(FALSE), mInModal(FALSE), 
		mSawClose(FALSE), mHwnd(NULL), mPrevWndProc(NULL)
{
}

CWindow::CWindow(HWND hWnd) : 

	mWeCreated(FALSE), mWeSubclassed(FALSE), mInModal(FALSE), 
		mSawClose(FALSE), mHwnd(NULL), mPrevWndProc(NULL)
{
	Attach(hWnd);
}

CWindow::~CWindow()
{
	try
	{
		if (WeCreated())
			Close();
		else if (WeSubclassed())
			UnSubclass();
	}
	catch (CComboError&) { }
}

CWindow& 
CWindow::operator=(HWND hWnd)
{
	Attach(hWnd);
	return *this;
}

void 
CWindow::Attach(HWND hWnd)
{
	pgpAssert(!WeCreated());
	pgpAssert(!WeSubclassed());

	// Allow NULL hWnds.
	if (IsntNull(hWnd))
	{
		pgpAssert(::IsWindow(hWnd));
	}

	mHwnd = hWnd;
}

void 
CWindow::Center() const
{
	// Center on parent, owner, or desktop window.
	PGPBoolean	isChild	= (GetStyle() & WS_CHILD ? TRUE : FALSE); 

	HWND	hwndCenter;

	if (isChild)
		hwndCenter = GetParent();
	else
		hwndCenter = GetWindow(GW_OWNER);

	if (hwndCenter == NULL)
		hwndCenter = GetDesktopWindow();

	CWindow	wndCenter(hwndCenter);

	// Don't center on minimized or invisible windows.
	PGPUInt32	wndCenterStyle	= wndCenter.GetStyle();

	if (!(wndCenterStyle & WS_VISIBLE) || (wndCenterStyle & WS_MINIMIZE))
		return;

	// Calculate coordinates.
	CRect	rcWindow;
	GetWindowRect(rcWindow);

	CRect	rcCenter;
	wndCenter.GetWindowRect(rcCenter);

	CRect	rcNew;

	rcNew.Left() = (rcCenter.Left() + rcCenter.Right()) / 2 - 
		rcWindow.Width() / 2;
	rcNew.Right() = rcNew.Left() + rcWindow.Width();

	rcNew.Top() = (rcCenter.Top() + rcCenter.Bottom()) / 2 - 
		rcWindow.Height() / 2;
	rcNew.Bottom() = rcNew.Top() + rcWindow.Height();

	if (isChild)
		wndCenter.ScreenToClient(rcNew);

	// Move the window.
	SetWindowPos(NULL, rcNew.Left(), rcNew.Top(), -1, -1, 
		SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE);
}

PGPBoolean 
CWindow::ClientToScreen(CRect& rect)
{
	CPoint	point(rect.Left(), rect.Top());
	CPoint	translatedPoint	= point;

	PGPBoolean	succeeded	= ClientToScreen(translatedPoint);

	if (succeeded)
	{
		rect.Offset(translatedPoint.X() - point.X(), 
			translatedPoint.Y() - point.Y());
	}

	return succeeded;
}

PGPBoolean 
CWindow::FitFormattedTextToWindow(
	const char	*formatString, 
	const char	*subString) const
{
	// Get dimensions of window.
	CRect	windowRect;
	GetWindowRect(windowRect);

	PGPUInt32	xWindow	= windowRect.Right() - windowRect.Left() - 3;

	// Get dimensions of text without the substring.
	CDC	wndDC;
	wndDC.AttachFromWindow(Handle());

	CDC	memDC;
	memDC.CreateCompatible(wndDC);

	HFONT	oldFont	= static_cast<HFONT>(memDC.SelectObject(
		GetStockObject(DEFAULT_GUI_FONT)));

	CString	displayStr;
	displayStr.Format(formatString, subString);

	CSize	messageSizeWSub	= memDC.GetTextExtentPoint32(displayStr);
	memDC.LPtoDP(messageSizeWSub);

	PGPUInt32	widthMessageWSub	= messageSizeWSub.CX();

	// Will we have to truncate?
	if (widthMessageWSub >= xWindow)
	{
		displayStr.Format(formatString, "");

		CSize	messageSizeWoSub	= memDC.GetTextExtentPoint32(displayStr);
		memDC.LPtoDP(messageSizeWoSub);

		PGPUInt32	widthMessageWoSub	= messageSizeWoSub.CX();

		if (widthMessageWoSub < xWindow)
		{
			// We only have to truncate the substring.
			CSize	substringSize	= memDC.GetTextExtentPoint32(subString);
			memDC.LPtoDP(substringSize);

			PGPUInt32	widthSubString	= substringSize.CX();
			PGPUInt32	xForSubstring	= xWindow - widthMessageWoSub;

			CString	truncedSubStr(subString);
			memDC.FitStringToWidth(truncedSubStr, xForSubstring);

			displayStr.Format(formatString, truncedSubStr.Get());
		}
		else
		{
			// We must truncate both format and substring combined.
			displayStr.Format(formatString, subString);
			memDC.FitStringToWidth(displayStr, xWindow);
		}
	}

	memDC.SelectObject(oldFont);
	memDC.Clear();
	wndDC.Clear();

	// Display the text.
	return SetWindowText(displayStr);
}

PGPBoolean 
CWindow::FitTextToWindow(const char *cstr) const
{
	// Get dimensions of window.
	CRect	windowRect;
	GetWindowRect(windowRect);

	// Get dimensions of text to be written
	CDC	wndDC;
	wndDC.AttachFromWindow(Handle());

	CDC	memDC;
	memDC.CreateCompatible(wndDC);

	HFONT	oldFont	= static_cast<HFONT>(memDC.SelectObject(
		GetStockObject(DEFAULT_GUI_FONT)));

	PGPUInt32	xWindow	= windowRect.Right() - windowRect.Left() - 3;

	CString	displayStr(cstr);
	memDC.FitStringToWidth(displayStr, xWindow);

	memDC.SelectObject(oldFont);
	memDC.Clear();
	wndDC.Clear();

	// Display the text.
	return SetWindowText(displayStr);
}

PGPBoolean 
CWindow::GetClientRect(RECT& rect) const
{
	return ::GetClientRect(Handle(), &rect);
}

PGPBoolean 
CWindow::GetWindowRect(RECT& rect) const
{
	return ::GetWindowRect(Handle(), &rect);
}

PGPBoolean 
CWindow::GetWindowText(CString& text) const
{
	PGPInt32	length	= GetWindowTextLength(Handle());
	return ::GetWindowText(Handle(), text.GetBuffer(length + 1), length + 1);
}

PGPInt32 
CWindow::MapWindowPoints(
	HWND		hwndTo, 
	LPPOINT		points, 
	PGPUInt32	numPoints) const
{
	PGPInt32	result	= ::MapWindowPoints(Handle(), hwndTo, points, 
		numPoints);

	if (result == 0)
		THROW_ERRORS(kPGPError_Win32WindowOpFailed, GetLastError());

	return result;
}

PGPBoolean 
CWindow::MoveWindow(
	PGPInt32	X, 
	PGPInt32	Y, 
	PGPInt32	width, 
	PGPInt32	height, 
	PGPBoolean	repaint) const
{
	return ::MoveWindow(Handle(), X, Y, width, height, repaint);
}

PGPBoolean 
CWindow::ScreenToClient(CRect& rect)
{
	CPoint	point(rect.Left(), rect.Top());
	CPoint	translatedPoint	= point;

	PGPBoolean	succeeded	= ScreenToClient(translatedPoint);

	if (succeeded)
	{
		rect.Offset(translatedPoint.X() - point.X(), 
			translatedPoint.Y() - point.Y());
	}

	return succeeded;
}

PGPBoolean 
CWindow::SetWindowText(const char *text) const
{
	pgpAssertStrValid(text);

	return (::SetWindowText(Handle(), text) == 0 ? FALSE : TRUE);
}

PGPBoolean 
CWindow::SetWindowPos(
	HWND		hWndInsertAfter, 
	PGPInt32	X, 
	PGPInt32	Y, 
	PGPInt32	cx, 
	PGPInt32	cy, 
	PGPUInt32	uFlags) const
{
	return (::SetWindowPos(Handle(), hWndInsertAfter, X, Y, cx, cy, 
		uFlags) ? TRUE : FALSE);
}

PGPBoolean 
CWindow::GetClassInfo(const char *className, WNDCLASSEX& classEx)
{
	pgpAssertStrValid(className);

	classEx.cbSize = sizeof(classEx);
	return GetClassInfoEx(UModuleInstance::Get(), className, &classEx);
}

ATOM 
CWindow::RegisterClass(const char *className, WNDCLASSEX& classEx)
{
	pgpAssertStrValid(className);

	classEx.cbSize			= sizeof(classEx);
	classEx.lpfnWndProc		= GlobalWindowProc;
	classEx.hInstance		= UModuleInstance::Get();
	classEx.lpszClassName	= className;

	ATOM	classAtom	= RegisterClassEx(&classEx);

	if (IsNull(classAtom))
		THROW_ERRORS(kPGPError_Win32WindowOpFailed, GetLastError());

	return classAtom;
}

ATOM 
CWindow::RegisterClass(
	const char	*className, 
	PGPUInt32	style, 
	PGPInt32	clsExtra, 
	PGPInt32	wndExtra, 
	HICON		icon, 
	HICON		smallIcon, 
	HCURSOR		cursor, 
	HBRUSH		background, 
	const char	*menuName)
{
	pgpAssertStrValid(className);

	WNDCLASSEX	classEx;

	classEx.style			= style;
	classEx.cbClsExtra		= clsExtra;
	classEx.cbWndExtra		= wndExtra;
	classEx.hIcon			= icon;
	classEx.hCursor			= cursor;
	classEx.hbrBackground	= background;
	classEx.lpszMenuName	= menuName;
	classEx.hIconSm			= smallIcon;

	ATOM	classAtom	= RegisterClass(className, classEx);

	if (IsNull(classAtom))
		THROW_ERRORS(kPGPError_Win32WindowOpFailed, GetLastError());

	return classAtom;
}

void 
CWindow::UnregisterClass(const char *classNameOrAtom)
{
	pgpAssert(IsntNull(classNameOrAtom));

	if (!::UnregisterClass(classNameOrAtom, UModuleInstance::Get()))
		THROW_ERRORS(kPGPError_Win32WindowOpFailed, GetLastError());
}

void 
CWindow::Subclass(HWND hWnd)
{
	pgpAssert(!WeCreated());
	pgpAssert(!WeSubclassed());
	pgpAssert(::IsWindow(hWnd));

	RefWindowMap();

	try
	{
		Attach(hWnd);

		mWeSubclassed	= TRUE;
		mInModal		= FALSE;
		mSawClose		= FALSE;

		mPrevWndProc = reinterpret_cast<WNDPROC>(SetWindowLong(GWL_WNDPROC, 
			reinterpret_cast<LONG>(GlobalWindowProcSubclassed)));

		if (IsNull(mPrevWndProc))
			THROW_ERRORS(kPGPError_Win32WindowOpFailed, GetLastError());

		AssocHandleToWindow(Handle(), this);
	}
	catch (CComboError&)
	{
		mWeSubclassed = FALSE;

		DerefWindowMap();
		throw;
	}
}

void 
CWindow::UnSubclass()
{
	pgpAssert(!WeCreated());
	pgpAssert(WeSubclassed());
	pgpAssert(IsntNull(mPrevWndProc));

	SetWindowLong(GWL_WNDPROC, reinterpret_cast<LONG>(mPrevWndProc));

	if (IsNull(mPrevWndProc))
		THROW_ERRORS(kPGPError_Win32WindowOpFailed, GetLastError());

	mPrevWndProc = NULL;
	mWeSubclassed = FALSE;
	mHwnd = NULL;

	DerefWindowMap();
}

void 
CWindow::ShowModal(HACCEL accel)
{
	pgpAssert(WeCreated());

	mInModal = TRUE;
	mSawClose = FALSE;

	// Enter simple modal loop.
	CWindow	owner(GetWindow(GW_OWNER));
	PGPBoolean	shouldReEnable	= FALSE;

	if (owner.IsWindow() && (owner.Handle() != CWindow::GetDesktopWindow()))
		shouldReEnable = !owner.EnableWindow(FALSE);

	try
	{
		ShowWindow(SW_SHOW);
		UpdateWindow();

		CAccelerator	caccel(accel);

		while (TRUE)
		{
			CMessage	msg;

			if (msg.Get() < 1)
				break;

			if (IsntNull(accel))
			{
				if (!caccel.Translate(Handle(), msg))
				{
					msg.Translate();
					msg.Dispatch();
				}
			}
			else
			{
				msg.Translate();
				msg.Dispatch();
			}

			if (mSawClose)
				break;
		}

		if (shouldReEnable)
		{
			ShowWindow(SW_HIDE);
			DestroyWindow();
			owner.EnableWindow(TRUE);
			owner.SetActiveWindow();
		}
		else
		{
			DestroyWindow();
		}
	}
	catch (CComboError&)
	{
		if (shouldReEnable)
			owner.EnableWindow(TRUE);
	}
}

void 
CWindow::Create(
	const char	*classNameOrAtom, 
	WNDPROC		superedProc, 
	const char	*windowName, 
	PGPUInt32	style, 
	PGPUInt32	exStyle, 
	PGPInt32	x, 
	PGPInt32	y, 
	PGPInt32	width, 
	PGPInt32	height, 
	HWND		parent, 
	HMENU		menu)
{
	pgpAssert(!WeCreated());
	pgpAssert(!WeSubclassed());
	pgpAssert(IsntNull(classNameOrAtom));

	RefWindowMap();

	try
	{
		mWeCreated	= TRUE;
		mInModal	= FALSE;
		mSawClose	= FALSE;

		mPrevWndProc = superedProc;

		HWND	hWnd	= CreateWindowEx(exStyle, classNameOrAtom, 
			windowName, style, x, y, width, height, parent, menu, 
			UModuleInstance::Get(), this);

		if (IsNull(hWnd))
			THROW_ERRORS(kPGPError_Win32WindowOpFailed, GetLastError());
	}
	catch (CComboError&)
	{
		mWeCreated = FALSE;

		DerefWindowMap();
		throw;
	}
}

void 
CWindow::Close()
{
	pgpAssert(WeCreated());
	SendMessage(WM_CLOSE);	// be nice
}

LRESULT 
CWindow::CallDefault()
{
	if (IsntNull(mPrevWndProc))	
	{
		return CallWindowProc(mPrevWndProc, Handle(), CurMsg().Message(), 
			CurMsg().WParam(), CurMsg().LParam());
	}
	else
	{
		pgpAssert(WeCreated());

		return DefWindowProc(Handle(), CurMsg().Message(), 
			CurMsg().WParam(), CurMsg().LParam());
	}
}

void 
CWindow::RefWindowMap()
{
	if (mWindowMapRefCount == 0)
	{
		mWindowMap = new CHashTable<CWindow>(WindowMapSize);
		mWindowMapLock = new CRMWOLock;
	}

	mWindowMapRefCount++;
}

void 
CWindow::DerefWindowMap()
{
	if ((mWindowMapRefCount > 0) && (--mWindowMapRefCount == 0))
	{
		delete mWindowMap;
		mWindowMap = NULL;

		delete mWindowMapLock;
		mWindowMapLock = NULL;
	}
}

CWindow * 
CWindow::HandleToWindow(HWND hWnd)
{
	pgpAssert(::IsWindow(hWnd));

	CWindow	*pWindow;
	mWindowMapLock->StartReading();

	try
	{
		pWindow	= mWindowMap->Lookup(reinterpret_cast<PGPUInt32>(hWnd));
	}
	catch (CComboError&)
	{
		mWindowMapLock->StopReading();
		throw;
	}

	mWindowMapLock->StopReading();
	return pWindow;
}

void 
CWindow::AssocHandleToWindow(HWND hWnd, CWindow *pWindow)
{
	pgpAssert(::IsWindow(hWnd));
	pgpAssertAddrValid(pWindow, CWindow);

	mWindowMapLock->StartWriting();

	try
	{
		mWindowMap->Add(reinterpret_cast<PGPUInt32>(hWnd), pWindow);
		pWindow->mHwnd = hWnd;
	}
	catch (CComboError&)
	{
		mWindowMapLock->StopWriting();
		throw;
	}

	mWindowMapLock->StopWriting();
}

void 
CWindow::UnAssocHandleFromWindow(HWND hWnd)
{
	pgpAssert(::IsWindow(hWnd));

	mWindowMapLock->StartWriting();

	try
	{
		mWindowMap->Remove(reinterpret_cast<PGPUInt32>(hWnd));
	}
	catch (CComboError&)
	{
		mWindowMapLock->StopWriting();
		throw;
	}

	mWindowMapLock->StopWriting();
}

PGPBoolean 
CWindow::DispatchCommon(LRESULT &result)
{
	result = 0;

	PGPUInt32	message	= CurMsg().Message();
	WPARAM		wParam	= CurMsg().WParam();
	LPARAM		lParam	= CurMsg().LParam();

	switch (message)
	{
	case WM_CHAR:
		OnChar(static_cast<char>(wParam), lParam);
		return TRUE;

	case WM_CLEAR:
		OnClear();
		return TRUE;

	case WM_CLOSE:
		OnClose();
		return TRUE;

	case WM_COMMAND:
		result = OnCommand(UMath::GetHighWord(wParam), 
			UMath::GetLowWord(wParam), reinterpret_cast<HWND>(lParam));
		return TRUE;

	case WM_CONTEXTMENU:
		OnContextMenu(reinterpret_cast<HWND>(wParam), 
			CPoint(UMath::GetLowWord(lParam), 
			UMath::GetHighWord(lParam)));
		return TRUE;

	case WM_COPY:
		OnCopy();
		return TRUE;

	case WM_CREATE:
		result = OnCreate(reinterpret_cast<LPCREATESTRUCT>(lParam));
		return TRUE;

	case WM_CTLCOLORSTATIC:
		result = reinterpret_cast<LRESULT>(OnCtlColorStatic(
			reinterpret_cast<HDC>(wParam), reinterpret_cast<HWND>(lParam)));
		return TRUE;

	case WM_CUT:
		OnCut();
		return TRUE;

	case WM_DESTROY:
		OnDestroy();
		return TRUE;

	case WM_DRAWITEM:
		result = OnDrawItem(wParam, 
			reinterpret_cast<LPDRAWITEMSTRUCT>(lParam));
		return TRUE;

	case WM_ERASEBKGND:
		result = OnEraseBkgnd(reinterpret_cast<HDC>(wParam));
		return TRUE;

	case WM_GETMINMAXINFO:
		OnGetMinMaxInfo(reinterpret_cast<LPMINMAXINFO>(lParam));
		return TRUE;

	case WM_HELP:
		OnHelp(reinterpret_cast<HELPINFO *>(lParam));
		return TRUE;

	case WM_INITMENU:
		OnInitMenu(reinterpret_cast<HMENU>(wParam));
		return TRUE;

	case WM_KEYDOWN:
		OnKeyDown(wParam, lParam);
		return TRUE;

	case WM_KEYUP:
		OnKeyUp(wParam, lParam);
		return TRUE;

	case WM_KILLFOCUS:
		OnKillFocus();
		return TRUE;

	case WM_LBUTTONDBLCLK:
		OnLButtonDblClk(wParam, CPoint(UMath::GetLowWord(lParam), 
			UMath::GetHighWord(lParam)));
		return TRUE;

	case WM_LBUTTONDOWN:
		OnLButtonDown(wParam, CPoint(UMath::GetLowWord(lParam), 
			UMath::GetHighWord(lParam)));
		return TRUE;

	case WM_LBUTTONUP:
		OnLButtonUp(wParam, CPoint(UMath::GetLowWord(lParam), 
			UMath::GetHighWord(lParam)));
		return TRUE;

	case WM_MEASUREITEM:
		result = OnMeasureItem(wParam, 
			reinterpret_cast<LPMEASUREITEMSTRUCT>(lParam));
		return TRUE;

	case WM_MENUSELECT:
		OnMenuSelect(UMath::GetLowWord(wParam), 
			UMath::GetHighWord(wParam), reinterpret_cast<HMENU>(lParam));
		return TRUE;

	case WM_MOUSEMOVE:
		OnMouseMove(wParam, CPoint(UMath::GetLowWord(lParam), 
			UMath::GetHighWord(lParam)));
		return TRUE;

	case WM_NCCREATE:
		result = OnNcCreate(reinterpret_cast<LPCREATESTRUCT>(lParam));
		return TRUE;

	case WM_NCDESTROY:
		OnNcDestroy();
		return TRUE;

	case WM_NOTIFY:
		result = OnNotify(wParam, reinterpret_cast<LPNMHDR>(lParam));
		return TRUE;

	case WM_PAINT:
		OnPaint(reinterpret_cast<HDC>(wParam));
		return TRUE;

	case WM_PASTE:
		OnPaste();
		return TRUE;

	case WM_SETFOCUS:
		OnSetFocus();
		return TRUE;

	case WM_SIZE:
		OnSize(wParam, 
			UMath::GetLowWord(lParam), 
			UMath::GetHighWord(lParam));
		return TRUE;

	case WM_SYSCOLORCHANGE:
		OnSysColorChange();
		return TRUE;

	case WM_SYSCOMMAND:
		result = OnSysCommand(wParam, UMath::GetLowWord(lParam), 
			UMath::GetHighWord(lParam));
		return TRUE;

	case WM_TIMER:
		OnTimer(wParam, reinterpret_cast<TIMERPROC *>(lParam));
		return TRUE;

	case WM_UNDO:
		result = OnUndo();
		return TRUE;

	default:
		return FALSE;
	}
}

LRESULT 
CWindow::WindowProc(const CMessage& msg)
{
	LRESULT	result	= 0;

	// Save old message.
	CMessage	oldMsg	= mCurMsg;
	mCurMsg = msg;

	try
	{
		// See if client wants to handle it.
		if (!PreProcess(CurMsg(), result) && !DispatchCommon(result))
			result = CallDefault();

		PostProcess(CurMsg(), result);
	}
	catch (CComboError& caughtError)
	{
		HandleWindowError(caughtError);
		result = 0;
	}

	// Restore old message.
	mCurMsg = oldMsg;

	return result;
}

LRESULT 
CALLBACK 
CWindow::GlobalWindowProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	try
	{
		if (uMsg == WM_NCCREATE)
		{
			// New window created, map window pointer.
			LPCREATESTRUCT	pCS	= reinterpret_cast<LPCREATESTRUCT>(lParam);

			CWindow	*pWindow	= static_cast<CWindow *>(pCS->lpCreateParams);
			pgpAssertAddrValid(pWindow, CWindow);

			AssocHandleToWindow(hWnd, pWindow);

			if (!pWindow->WindowProc(CMessage(uMsg, wParam, lParam, hWnd)))
			{
				UnAssocHandleFromWindow(hWnd);
				return FALSE;
			}

			return TRUE;
		}
		else
		{
			// Find CWindow pointer corresponding to this HWND.
			CWindow	*pWindow	= HandleToWindow(hWnd);

			if (IsNull(pWindow))
				return DefWindowProc(hWnd, uMsg, wParam, lParam);

			LRESULT	result	= pWindow->WindowProc(CMessage(uMsg, wParam, 
				lParam, hWnd));

			// Unmap window handle on destruction.
			if (uMsg == WM_NCDESTROY)
			{
				UnAssocHandleFromWindow(hWnd);
				DerefWindowMap();
				pWindow->mWeCreated = FALSE;

				result = 0;
			}

			return result;
		}
	}
	catch (CComboError&)
	{
		pgpAssert(FALSE);
		return DefWindowProc(hWnd, uMsg, wParam, lParam);
	}
}

LRESULT 
CALLBACK 
CWindow::GlobalWindowProcSubclassed(
	HWND	hWnd, 
	UINT	uMsg, 
	WPARAM	wParam, 
	LPARAM	lParam)
{
	try
	{
		// Find CWindow pointer corresponding to this HWND;
		CWindow	*pWindow	= HandleToWindow(hWnd);

		if (IsNull(pWindow))
			THROW_PGPERROR(kPGPError_ItemNotFound);

		LRESULT	result	= pWindow->WindowProc(CMessage(uMsg, wParam, lParam, 
			hWnd));

		// Unmap window handle on destruction.
		if (uMsg == WM_NCDESTROY)
		{
			UnAssocHandleFromWindow(hWnd);
			pWindow->UnSubclass();

			result = 0;
		}
		
		return result;
	}
	catch (CComboError&)
	{
		pgpAssert(FALSE);
		return DefWindowProc(hWnd, uMsg, wParam, lParam);
	}
}

void 
CWindow::OnClose() 
{
	if (mInModal)
	{
		mInModal = FALSE;
		mSawClose = TRUE;
	}
	else
	{
		CallDefault();
	}
}

BOOL 
CWindow::OnCommand(PGPUInt16 notifyCode, PGPUInt16 itemId, HWND ctrl)
{
	// Reflect message to child control.
	if (ctrl != Handle())
	{
		BOOL	result	= CallDefault();

		CWindow	ctrlWnd(ctrl);

		ctrlWnd.SendMessage(WM_COMMAND, UMath::MakeDWord(itemId, 
			notifyCode), reinterpret_cast<LPARAM>(ctrl));

		return result;
	}
	else
	{
		return 0;
	}
}

BOOL 
CWindow::OnDrawItem(PGPUInt16 ctrlId, LPDRAWITEMSTRUCT pDIS)
{
	// Reflect message to child control.
	CWindow	ctrlWnd(pDIS->hwndItem);

	if (!(pDIS->CtlType & ODT_MENU) && (ctrlWnd.Handle() != Handle()))
	{
		BOOL	result	= CallDefault();

		ctrlWnd.SendMessage(WM_DRAWITEM, ctrlId, reinterpret_cast<LPARAM>(
			pDIS));

		return result;
	}
	else
	{
		return FALSE;
	}
}
	
BOOL 
CWindow::OnMeasureItem(PGPUInt16 ctrlId, LPMEASUREITEMSTRUCT pMIS)
{	
	// Reflect message to child control.
	CWindow	ctrlWnd(GetDlgItem(ctrlId));

	if (!(pMIS->CtlType & ODT_MENU) && (ctrlWnd.Handle() != Handle()))
	{
		BOOL	result = CallDefault();

		ctrlWnd.SendMessage(WM_MEASUREITEM, ctrlId, 
			reinterpret_cast<LPARAM>(pMIS));

		return result;
	}
	else
	{
		return FALSE;
	}
}

PGPUInt32 
CWindow::OnNotify(PGPUInt16 ctrlId, LPNMHDR pNMHDR)
{
	// Reflect message to child control.
	CWindow	ctrlWnd(pNMHDR->hwndFrom);

	if (ctrlWnd.Handle() != Handle())
	{
		PGPUInt32	result	= CallDefault();

		ctrlWnd.SendMessage(WM_NOTIFY, ctrlId, reinterpret_cast<LPARAM>(
			pNMHDR));

		return result;
	}
	else
	{
		return 0;
	}
}
